<?php

/**
 * Handles configuration and CRUD functions
 */
class AdminModel
{

    /**
     * get all formulas
     * parameters none
     * returns JSON string
     */
     public static function getFormulas($vehicleid,$id=-1,$encodeentities=true)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $names=array();
        $results=array();
        $sql="SELECT DISTINCT name FROM formulas WHERE vehicleid=:vehicleid ";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        while ($row =$query->fetch(PDO::FETCH_ASSOC)){
           $names[]=$row['name'];
        }
        foreach($names as $name){
           $sql="SELECT * FROM formulas WHERE name=:name AND vehicleid=:vehicleid ";
           $query = $database->prepare($sql);
           $parameters = array(':name' => $name,':vehicleid' => $vehicleid);
           $query->execute($parameters);
           //$results[$name]=$query->fetchAll(PDO::FETCH_ASSOC);
           $formulas=$query->fetchAll(PDO::FETCH_ASSOC);
           $results[$name]=array();
           foreach($formulas as $formula){
               $selected=($formula['id']==$id)?true:false;
               $_formula=$encodeentities ? htmlentities(Encryption::decrypt($formula['b_formula'])):Encryption::decrypt($formula['b_formula']);
               $results[$name][]=array('id' => $formula['id'],'description' => $formula['description'],'name' => $formula['name'],'formula' => $_formula,'selected' => $selected);
           }
           //$results[$name]=array('id' => $formula['id'],'name' => $formula['name'],'formula' => Encryption::decrypt($formula['b_formula']));
        }
        //return json_encode($results,JSON_PRETTY_PRINT);
        return $results;
     }

    /**
     * get a formula based on given id
     * parameters id(int)
     * returns JSON string
     */
     public static function getFormulaById($id){

           $database = DatabaseFactory::getFactory()->getConnection();

           $sql="SELECT * FROM formulas WHERE id=:id ";
           $query = $database->prepare($sql);
           $parameters = array(':id' => $id);
           $query->execute($parameters);
           $results=$query->fetch(PDO::FETCH_ASSOC);
           //change results array to include encrypted formula
           $results['formula']=Encryption::decrypt($results['b_formula']);
           return $results;
     }


     /**
     * save a formula
     * parameters id(int),vehicleid(int),formula(string),description(string),usedon(string)
     * returns array
     */
     public static function saveFormula($id,$vehicleid,$formula,$description,$usedon)
     {
        $database = DatabaseFactory::getFactory()->getConnection();
        $encryptedFormula=Encryption::encrypt($formula);
        if($id>0){
           $sql="UPDATE formulas SET b_formula=:formula,description=:description,name=:usedon WHERE id=:id;";
           $query = $database->prepare($sql);
           $parameters=array(':formula' => $encryptedFormula, ':description' => $description, ':usedon' => $usedon, ':id' => $id);
           $query->execute($parameters);
        }
        //or add a new  record
        else{
           $sql="INSERT INTO formulas (id,vehicleid,b_formula,description,name) VALUES (NULL,:vehicleid,:formula,:description,:usedon)";
           $query = $database->prepare($sql);
           $parameters=array(':vehicleid' => $vehicleid,':formula' => $encryptedFormula, ':description' => $description, ':usedon' => $usedon);
            //echo '<br><br><br>[ PDO DEBUG ]: ' . Helper::debugPDO($sql, $parameters);  exit();
           $query->execute($parameters);
           $id=$database->lastInsertId();
        }
        $results=self::getFormulas($vehicleid,$id);
        return $results;

     }

     /*
     * delete a formula
     * parameters id(int)
     * returns array
     */
     public static function deleteFomula($id,$vehicleid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        if($id>0){
           $sql="DELETE FROM formulas WHERE id=:id;";
           $query = $database->prepare($sql);
           $parameters=array(':id' => $id);
           $query->execute($parameters);
        }
        $results=self::getFormulas($vehicleid);
        return $results;
     }



    /**
     * get a view based on given parameters
     * parameters vehicleid(int),schedule(int),type(string),view(string),version(int)
     * returns JSON string
     */
     public static function getView($vehicleid,$schedule,$type,$view,$version)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $releasestatus=0;
        $rcmversion=0;
        $inspectionversions=array();
        $latestversion=0;
        $json="";
        $results="";
        //need to get latest version if version is not given
        if($version==""||$version=="null"){
           $version=HmsModel::getLatestInspectionVersion($vehicleid);
        }
        //get the available versions for the selected vehicle
        $sql="SELECT * FROM inspectionversion WHERE vehicleid=:vehicleid ORDER BY created;";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
           $json='{"version":'.$row['version'].',"created":"'.$row['created'].'","selected":';
           $json.=(($row['version']==$version)?'true':'false').'}';
           $inspectionversions[]=$json;
        }
        //get the status from inspectionversion
        $sql="SELECT status,rcmversion FROM inspectionversion WHERE vehicleid=:vehicleid AND version=:version";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid,':version'=>$version);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
          $rcmversion=$row['rcmversion'];
          $releasestatus=$row['status'];
        }

        //if the status is pre-release then the master (full inspection) is not available
        if($releasestatus==2&&$schedule==1)$schedule=2;


        //first try and select a saved version of the view for this schedule and version
        $sql="SELECT name,image,b_poddata FROM view WHERE name=:view AND type=:type AND vehicleid=:vehicleid AND scheduleid=:schedule AND version=:version";
        $query = $database->prepare($sql);
        $parameters = array(':view'=>$view,':type'=>$type,':vehicleid' => $vehicleid,':schedule'=>$schedule,':version'=>$version);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
              //$results='{"versions":['.join(',',$inspectionversions).'],"status":'.$releasestatus.',"rcm":'.$rcmversion.',"name":"'.$row['name'].'","image":"'.$row['image'].'","pods":'.stripslashes(str_replace(array("({","})","http://www.i-xpress.co.uk/hmspit/"),array("{","}",""), $row['poddata'])).',"saved":true}';
              $results='{"versions":['.join(',',$inspectionversions).'],"status":'.$releasestatus.',"rcm":'.$rcmversion.',"name":"'.$row['name'].'","image":"'.$row['image'].'","pods":'.stripslashes(str_replace(array("({","})","http://www.i-xpress.co.uk/hmspit/"),array("{","}",""), encryption::decrypt($row['b_poddata']))).',"saved":true}';
        }
        //if no saved version exists, select the master view (full inspection) for this version
        if($results==""){
           $sql="SELECT name,image,b_poddata FROM view WHERE name=:view AND type=:type AND vehicleid=:vehicleid AND scheduleid='1' AND version=:version";
           $query = $database->prepare($sql);
           $parameters = array(':view'=>$view,':type'=>$type,':vehicleid' => $vehicleid,':version'=>$version);
           $query->execute($parameters);
           while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
              //$results='{"versions":['.join(',',$inspectionversions).'],"status":'.$releasestatus.',"rcm":'.$rcmversion.',"name":"'.$row['name'].'","image":"'.$row['image'].'","pods":'.stripslashes(str_replace(array("({","})","http://www.i-xpress.co.uk/hmspit/"),array("{","}",""), $row['poddata'])).',"saved":false}';
              $results='{"versions":['.join(',',$inspectionversions).'],"status":'.$releasestatus.',"rcm":'.$rcmversion.',"name":"'.$row['name'].'","image":"'.$row['image'].'","pods":'.stripslashes(str_replace(array("({","})","http://www.i-xpress.co.uk/hmspit/"),array("{","}",""), encryption::decrypt($row['b_poddata']))).',"saved":false}';
           }
           //Finally,if no saved master version exists, just return a blank view
           if($results==""){
              $results='{"versions":['.join(',',$inspectionversions).'],"status":'.$releasestatus.',"rcm":'.$rcmversion.',"name":"'.$view.'","image":"img/configuration/blank.jpg","pods":[],"saved":false}';
           }
        }
        return $results;
     }

    /**
     * get a list of repair schemes based for a given vehicle
     * parameters vehicleid(int),selected(int)
     * returns array
     */
     public static function getRepairSchemes($vehicleid,$selected=-1)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $results=array();
        $results[]=array('id'=>-1,'reference'=>'New Record','selected' =>'false');
        $sql="SELECT * FROM repairschemes WHERE vehicleid=:vehicleid";
        $query = $database->prepare($sql);
        $parameters=array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
           $isselected=($row['id']==$selected)?'true':'false';
           $results[]=array('id'=>$row['id'],'reference'=>$row['reference'],'selected' =>$isselected);
        }
        return $results;
     }

    /**
     * get a list of RCM versions based for a given vehicle
     * parameters vehicleid(int),descending(bool)
     * returns array
     */
     public static function getRCMVersions($vehicleid,$descending=false)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM rcmversion WHERE vehicleid=:vehicleid";
        if($descending)$sql.=" ORDER BY version DESC";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        return $query->fetchAll(PDO::FETCH_ASSOC);
     }

    /**
     * get a list of all vehicles
     * parameters none
     * returns array
     */
     public static function getAllVehicles()
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql = "SELECT id,name FROM vehicle ORDER BY id";
        $query = $database->prepare($sql);
        $query->execute();
        $results=$query->fetchAll(PDO::FETCH_ASSOC);
        return $results;
     }

    /**
     * get a list of all roles
     * parameters none
     * returns array
     */
     public static function getAllRoles()
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM roles";
        $query = $database->prepare($sql);
        $query->execute();
        return $query->fetchAll(PDO::FETCH_ASSOC);
     }

    /**
     * get RCM information for a given vehicle and version
     * parameters vehicleid(int),version(int)
     * returns array
     */
     public static function getRCMInfo($vehicleid,$version)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $results=array();
        $sql="SELECT id,reference,description,data FROM rcm WHERE vehicleid=:vehicleid AND version=:version ORDER BY reference";
        $query = $database->prepare($sql);
        $parameters = array(':version' => $version,':vehicleid'=>$vehicleid);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
           $decoded=json_decode($row['data']);
           if($decoded&&strtoupper($decoded->RCM55)=='Y'){
              $results[]=array("id"=>$row['id'],"reference"=>$row['reference'],"description"=>$row['description']);
           }
         }
        return $results;
     }

    /**
     * get SSI descriptions for a given vehicle
     * parameters vehicleid(int)
     * returns array
     */
     public static function getSSIDescriptions($vehicleid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM ssiid WHERE vehicleid=:vehicleid";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        return $query->fetchAll(PDO::FETCH_ASSOC);

     }

    /**
     * get RCM record for a given id
     * parameters rcmid(int)
     * returns array
     */
     public static function getRCMRecord($rcmid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT data FROM rcm WHERE id=:rcmid";
        $query = $database->prepare($sql);
        $parameters = array(':rcmid' => $rcmid);
        $query->execute($parameters);
        return $query->fetch()->data;
     }
    /**
     * get a list of RCM records for a given vehicle,version an filter
     * parameters vehicleid(int),version(int),filtermap(int)
     * returns JSON string
     */
     public static function getRCMByFilter($vehicleid,$version,$filtermap)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $filters=array();
        $filters[1]=array("RCM"=>"RCM55","condition"=>"Y");
        $filters[2]=array("RCM"=>"RCM55","condition"=>"N");
        $filters[3]=array("RCM"=>"RCM20","condition"=>"Scheduled");
        $filters[4]=array("RCM"=>"RCM20","condition"=>"Non-scheduled");
        $filters[5]=array("RCM"=>"RCM11","condition"=>"FD");
        $filters[6]=array("RCM"=>"RCM11","condition"=>"ED");
        $filters[7]=array("RCM"=>"RCM11","condition"=>"AD");
        $filters[8]=array("RCM"=>"RCM17","condition"=>"A");
        $filters[9]=array("RCM"=>"RCM17","condition"=>"B");
        $filters[10]=array("RCM"=>"RCM17","condition"=>"C");
        $filters[11]=array("RCM"=>"RCM17","condition"=>"D");
        $filters[12]=array("RCM"=>"RCM14","condition"=>"H");
        $filters[13]=array("RCM"=>"RCM38","condition"=>"H");
        $filters[14]=array("RCM"=>"RCM13","condition"=>"S");
        $filters[15]=array("RCM"=>"RCM13","condition"=>"O");
        $filters[16]=array("RCM"=>"RCM13","condition"=>"NOP");
        $filters[17]=array("RCM"=>"RCM23","condition"=>"1");
        $filters[18]=array("RCM"=>"RCM23","condition"=>"2");
        $filters[19]=array("RCM"=>"RCM23","condition"=>"3");
        $filters[20]=array("RCM"=>"RCM23","condition"=>"4");
        $filters[21]=array("RCM"=>"RCM29","condition"=>"Quantitative");
        $filters[22]=array("RCM"=>"RCM29","condition"=>"Qualitative");
        $filters[23]=array("RCM"=>"RCM29","condition"=>"Max dimension");
        $filters[24]=array("RCM"=>"RCM19","condition"=>"Time");
        $filters[25]=array("RCM"=>"RCM19","condition"=>"Use");
        $locked=0;
        $rcmversion=0;
        $rcmcreated="";
        $forecastUsage=0;
        $nomRefStartDate="";
        $data=array();
        $json='{';
        $vals=array();
        $sql="SELECT version,created,locked,forecastUsage,nomRefStartDate FROM rcmversion WHERE vehicleid=:vehicleid AND version=:version";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid'=>$vehicleid,':version' => $version);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
           $locked=$row['locked'];
           $rcmversion=$row['version'];
           $rcmcreated=$row['created'];
           $forecastUsage=$row['forecastUsage'];
           $nomRefStartDate=$row['nomRefStartDate'];
        }
        $sql="SELECT id,reference,description,data FROM rcm WHERE vehicleid=:vehicleid AND version=:version ORDER BY reference";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid'=>$vehicleid,':version' => $version);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
           $vals=json_decode($row['data'],true);
           if($filtermap=='0'){
              $data[]='{"id":'.$row['id'].',"reference":"'.$row['reference'].'","description":"'.$row['description'].'"}';
           }
           else{
              if($vals[$filters[$filtermap]['RCM']]==$filters[$filtermap]['condition']){
                 $data[]='{"id":'.$row['id'].',"reference":"'.$row['reference'].'","description":"'.$row['description'].'"}';
              }
           }
        }
        $json.='"version":'.$rcmversion.',"created":"'.$rcmcreated.'","forecastUsage":"'.$forecastUsage.'","nomRefStartDate":"'.$nomRefStartDate.'","locked":'.$locked.',"records":['.join(',',$data).']}';
        return $json;
     }

     /**
     * get an RCM record for a given id
     * parameters id(int)
     * returns JSON string
     */
     public static function openRCMRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $json="{";
        $version=0;
        $reference="";
        if($id){
           $sql="SELECT rcm.id,rcm.reference,rcm.data,rcmversion.locked,rcmversion.version FROM rcm,rcmversion WHERE rcm.version=rcmversion.version AND rcm.id=:id";
           $query = $database->prepare($sql);
           $parameters = array(':id'=>$id);
           $query->execute($parameters);
           while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
              $json.= '"id":'.$row['id'].',"locked":'.$row['locked'].',"data":'.str_replace(array("\n","\r")," ",$row['data']);
              $reference=$row['reference'];
              $version=$row['version']-1;//previous version
           }
           //get the previous version for comparison
           $sql="SELECT rcm.data,rcmversion.version,rcmversion.created FROM rcm,rcmversion WHERE rcm.version=rcmversion.version AND rcm.version=:version AND rcm.reference=:reference";
           $query = $database->prepare($sql);
           $parameters = array(':version'=>$version,':reference'=>$reference);
           $query->execute($parameters);
           while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
              $json.= ',"prevversion":'.$row['version'].',"prevcreated":"'.$row['created'].'","prevdata":'.str_replace(array("\n","\r")," ",$row['data']);
           }
        }
        $json.="}";
        return $json;
     }

     /**
     * save an RCM record to the database
     * parameters vehicleid(int),version(int),id(int),reference(string),description(string),data(string)
     * returns int
     */
     public static function saveRCMRecord($vehicleid,$version,$id,$reference,$description,$data)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        if($id){
           if($id>0){
              $sql="UPDATE rcm SET reference=:reference,description=:description,data=:data WHERE id=:id";
              $query = $database->prepare($sql);
              $parameters = array(':reference' => $reference, ':description' => $description,':data' => $data, ':id'=>$id);
              $query->execute($parameters);
           }
           else{
              $sql="INSERT INTO rcm(id, version, vehicleid, reference, description, data) VALUES (null,:version,:vehicleid,:reference,:description,:data)";
              $query = $database->prepare($sql);
              $parameters = array(':version' => $version,':vehicleid' => $vehicleid,':reference' => $reference, ':description' => $description,':data' => $data);
              $query->execute($parameters);
              $id = $database->lastInsertId();
           }
        }
        return $id;
     }

     /**
     * delete an RCM record from the database
     * parameters id(int)
     * returns none
     */
     public static function deleteRCMRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        if($id){
           $sql="DELETE FROM rcm WHERE id=:id";
           $query = $database->prepare($sql);
           $parameters = array(':id'=>$id);
           $query->execute($parameters);
        }
     }

     /**
     * create a new RCM version
     * parameters vehicleid(int),copyversion(int),forecastusage(int),nomstartdate(string)
     * returns string
     */
     public static function createRCMVersion($vehicleid,$copyversion,$forecastusage,$nomrefstartdate)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $newversion=0;
        //firstly increment the max version for this vehicle id
        $sql="SELECT MAX(version) AS 'latestversion' FROM rcmversion WHERE vehicleid=:vehicleid";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        $newversion=$query->fetch()->latestversion+1;
        //and add it to the versions table
        $sql="INSERT INTO rcmversion(id, version, vehicleid, created,forecastUsage,nomRefStartDate) VALUES (null,:newversion,:vehicleid,null,:forecastusage,:nomrefstartdate)";
        $query = $database->prepare($sql);
        $parameters = array(':newversion' => $newversion,':vehicleid' => $vehicleid,':forecastusage' => $forecastusage,':nomrefstartdate' => $nomrefstartdate);
        $query->execute($parameters);
        $id = $database->lastInsertId();
        //now if required, copy the records from the specified version
        if($copyversion>0){
           $sql="CREATE TEMPORARY TABLE tmp SELECT * from rcm WHERE vehicleid=:vehicleid AND version=:version";
           $query = $database->prepare($sql);
           $parameters = array(':vehicleid' => $vehicleid,':version' => $copyversion);
           $query->execute($parameters);
           // drop autoincrement field
           $sql="ALTER TABLE tmp drop id;";
           $query = $database->prepare($sql);
           $query->execute();
           $sql="UPDATE tmp SET version=:newversion";
           $query = $database->prepare($sql);
           $parameters = array(':newversion' => $newversion);
           $query->execute($parameters);
           $sql="INSERT INTO rcm SELECT 0,tmp.* FROM tmp";
           $query = $database->prepare($sql);
           $query->execute();
           $sql="DROP TABLE tmp";
           $query = $database->prepare($sql);
           $query->execute();
        }
        return "RCM version ".$id." created";
     }

     /**
     * lock the given RCM version
     * parameters vehicleid(int),version(int)
     * returns string
     */
     public static function lockRCMVersion($vehicleid,$version)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $locked=1;
        $sql="UPDATE rcmversion SET locked=:locked WHERE vehicleid=:vehicleid AND version=:version;";
        $query = $database->prepare($sql);
        $parameters = array(':locked' => $locked,':vehicleid' => $vehicleid,':version' => $version);
        try{
           $query->execute($parameters);
           return "RCM version ".$version." has been locked";
        }
        catch (PDOException $e){
           return "The transaction failed: " .$e->getMessage() .'<br>';
        }
        catch (Exception $e) {
           return "The transaction failed: ".$e->getMessage() .'<br>';
        }

     }

     /**
     * edit an inspection version
     * parameters vehicleid(int),version(int),description(string),status(int)
     * returns boolean
     */
     public static function editInspectionVersion($vehicleid,$version,$description,$status)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="UPDATE inspectionversion SET description=:description,status=:status WHERE vehicleid=:vehicleid AND version=:version";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid,':version' => $version,':description' => $description, ':status' => $status);
        if($query->execute($parameters))return true;
        else return false;
     }

     /**
     * get an inspection version
     * parameters vehicleid(int),version(int)
     * returns array
     */
     public static function getInspectionVersion($vehicleid,$version)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT version,created,description, CASE status
           WHEN 1 then 'draft'
           WHEN 2 then 'pre-release'
           WHEN 3 then 'release'
           ELSE 'unknown'
        END as 'status' FROM inspectionversion WHERE vehicleid=:vehicleid AND version=:version";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid,':version' => $version);
        $query->execute($parameters);
        return $query->fetch(PDO::FETCH_ASSOC);
     }

     /**
     * create an inspection version
     * parameters vehicleid(int),description(string),rcmversion(int)
     * returns int
     */
     public static function createInspectionVersion($vehicleid,$description,$rcmversion)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $newversion=0;
        $previousversion=0;
        //increment inspection version for this vehicle id
        //firstly increment the max version for this vehicle id
        $sql="SELECT MAX(version) AS 'latestversion' FROM inspectionversion WHERE vehicleid=:vehicleid";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid);
        $query->execute($parameters);
        while ($row = $query->fetch(PDO::FETCH_ASSOC)) {
            $previousversion=$row['latestversion'];
            $newversion=$row['latestversion']+1;
        }
        //and add it to the versions table
        $sql="INSERT INTO inspectionversion(id, version, vehicleid, rcmversion, created,description, status) VALUES (null,:newversion,:vehicleid,:rcmversion,null,:description,1)";
        $query = $database->prepare($sql);
        $parameters = array(':newversion' => $newversion,':vehicleid' => $vehicleid,':rcmversion' => $rcmversion,':description' => $description);
        $query->execute($parameters);
        $id=$database->lastInsertId();

        //create master views
        //select previous version of views into temporary table
        $sql="CREATE TEMPORARY TABLE tmp SELECT * from view WHERE vehicleid=:vehicleid AND scheduleid=1 AND version=:previousversion";
        $query = $database->prepare($sql);
        $parameters = array(':vehicleid' => $vehicleid,':previousversion' => $previousversion);
        $query->execute($parameters);

        // drop autoincrement field
        $sql="ALTER TABLE tmp drop Id";
        $query = $database->prepare($sql);
        $query->execute();

        //update with new inspection version
        $sql="UPDATE tmp SET version=:newversion";
        $query = $database->prepare($sql);
        $parameters = array(':newversion' => $newversion);
        $query->execute($parameters);
        //append tmp to table view
        $sql="INSERT INTO view SELECT 0,tmp.* FROM tmp";
        $query = $database->prepare($sql);
        $query->execute();
        //and delete tmp
        $sql="DROP TABLE tmp";
        $query = $database->prepare($sql);
        $query->execute();
        return $newversion;
     }

     /**
     * save an inspection view
     * parameters vehicleid(int),scheduleid(int),view(string),image(string),metriccount(int),pods(string),version(int)
     * returns string
     */
     public static function saveView($vehicleid,$scheduleid,$view,$image,$metriccount,$pods,$version)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM view WHERE name=:view AND vehicleid=:vehicleid AND scheduleid=:scheduleid AND version=:version";
        $query = $database->prepare($sql);
        $parameters = array(':view' => $view, ':vehicleid' => $vehicleid, ':scheduleid' => $scheduleid, ':version' => $version);
        $query->execute($parameters);
        if($query->fetch(PDO::FETCH_NUM) > 0){
           $sql="UPDATE view SET image=:image,metriccount=:metriccount,b_poddata=:pods WHERE name=:view AND vehicleid=:vehicleid AND scheduleid=:scheduleid AND version=:version";
           $query = $database->prepare($sql);
           $parameters = array(':image' => $image, ':metriccount' => $metriccount, ':pods' => Encryption::encrypt($pods), ':view' => $view, ':vehicleid' => $vehicleid, ':scheduleid' => $scheduleid, ':version' => $version);
           $query->execute($parameters);
        }
        else{
           $sql="INSERT INTO view (id,vehicleid,scheduleid,version,type,name,image,metriccount,b_poddata) VALUES (NULL,:vehicleid,:scheduleid,:version,'repair',:view,:image,:metriccount,:pods)";
           $query = $database->prepare($sql);
           $parameters = array(':image' => $image, ':metriccount' => $metriccount, ':pods' => Encryption::encrypt($pods), ':view' => $view, ':vehicleid' => $vehicleid, ':scheduleid' => $scheduleid, ':version' => $version);
           $query->execute($parameters);
        }
        return "saved";
     }

     /**
     * delete a vehicle info record
     * parameters id(int)
     * returns boolean
     */
     public static function deleteVehicleInfoRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="DELETE FROM vehicleinfo WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters=array(':id' => $id);
        if($query->execute($parameters))return true;
        else return false;
     }

     /**
     * save a vehicle info record
     * parameters birthcert(string),borndate(date),vrn(string),buildstandard(string),vehiclerole(string),recordid(int)
     * returns none
     */
     public static function saveVehicleInfoRecord($birthcert,$borndate,$vrn,$buildstandard,$vehiclerole,$recordid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        //update if existing
        if($recordid>0){
           $sql="UPDATE vehicleinfo SET latestbirthcertificate=:birthcert,borndate=:borndate,vrn=:vrn,buildstandard=:buildstandard,vehiclerole=:vehiclerole WHERE id=:recordid";
           $query = $database->prepare($sql);
           $parameters=array(':birthcert' => $birthcert,':borndate' => $borndate, ':vrn' => $vrn,':buildstandard' => $buildstandard, ':vehiclerole' => $vehiclerole, ':recordid' => $recordid);
           $query->execute($parameters);
        }
        //or add a new  record
        else{
           $sql="INSERT INTO vehicleinfo (id,latestbirthcertificate,borndate,vrn,buildstandard,vehiclerole) VALUES (NULL,:birthcert,:borndate,:vrn,:buildstandard,:vehiclerole)";
           $query = $database->prepare($sql);
           $parameters=array(':birthcert' => $birthcert,':borndate' => $borndate, ':vrn' => $vrn,':buildstandard' => $buildstandard, ':vehiclerole' => $vehiclerole);
           $query->execute($parameters);
        }

     }

     /**
     * delete a reference material record
     * parameters id(int)
     * returns boolean
     */
     public static function deleteReferenceMaterialRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="DELETE FROM referencematerial WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters=array(':id' => $id);
        if($query->execute($parameters))return true;
        else return false;
     }

     /**
     * save a reference material record
     * parameters document(string),title(string),vrn(string),vidno(string),reference(string),issuedate(string),organisation(string),recordid(int)
     * returns none
     */
     public static function saveReferenceMaterialRecord($document,$title,$vrn,$vidno,$reference,$issuedate,$organisation,$recordid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        //update if existing
        if($recordid>0){
           $sql="UPDATE referencematerial SET document=:document,documenttitle=:title,vrn=:vrn,vidno=:vidno,reference=:reference,issuedate=:issuedate,organisationname=:organisation WHERE id=:recordid;";
           $query = $database->prepare($sql);
           $parameters=array(':document' => $document,':title' => $title, ':vrn' => $vrn,':vidno' => $vidno, ':reference' => $reference, ':issuedate' => $issuedate, ':organisation' => $organisation, ':recordid' => $recordid);
           $query->execute($parameters);
        }
        //or add a new  record
        else{
           $sql="INSERT INTO referencematerial (id,document,documenttitle,vrn,vidno,reference,issuedate,organisationname) VALUES (NULL,:document,:title,:vrn,:vidno,:reference,:issuedate,:organisation)";
           $query = $database->prepare($sql);
           $parameters=array(':document' => $document,':title' => $title, ':vrn' => $vrn,':vidno' => $vidno, ':reference' => $reference, ':issuedate' => $issuedate, ':organisation' => $organisation);
           $query->execute($parameters);
        }
     }

     /**
     * delete a minutes record
     * parameters id(int)
     * returns boolean
     */
     public static function deleteMinutesRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="DELETE FROM minutes WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters=array(':id' => $id);
        if($query->execute($parameters))return true;
        else return false;
     }

     /**
     * save a minutes record
     * parameters document(string),title(string),reference(string),issuedate(string),organisation(string),recordid(int)
     * returns none
     */
     public static function saveMinutesRecord($document,$title,$reference,$issuedate,$organisation,$recordid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        //update if existing
        if($recordid>0){
           $sql="UPDATE minutes SET document=:document,reporttitle=:title,reference=:reference,issuedate=:issuedate,organisation=:organisation WHERE id=:recordid;";
           $query = $database->prepare($sql);
           $parameters=array(':document' => $document,':title' => $title, ':reference' => $reference, ':issuedate' => $issuedate, ':organisation' => $organisation, ':recordid' => $recordid);
           $query->execute($parameters);
        }
        //or add a new  record
        else{
           $sql="INSERT INTO minutes (id,document,reporttitle,reference,issuedate,organisation) VALUES (NULL,:document,:title,:reference,:issuedate,:organisation)";
           $query = $database->prepare($sql);
           $parameters=array(':document' => $document,':title' => $title, ':reference' => $reference, ':issuedate' => $issuedate, ':organisation' => $organisation);
           $query->execute($parameters);
        }
     }

     /**
     * open a repair scheme record
     * parameters did(int)
     * returns array
     */
     public static function openRepairSchemeRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM repairschemes WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters=array(':id' => $id);
        $query->execute($parameters);
        return $query->fetchAll();

     }

     /**
     * save a minutes record
     * parameters id(int),vehicleid(int),title(string),reference(string),data(string),description(string),vrn(string),type(string)
     * returns array
     */
     public static function saveRepairSchemeRecord($id,$vehicleid,$title,$reference,$data,$description,$vrn,$type)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        if($id>0){
           $sql="UPDATE repairschemes SET title=:title,reference=:reference,data=:data,description=:description,vrn=:vrn,type=:type WHERE id=:id;";
           $query = $database->prepare($sql);
           $parameters=array(':title' => $title, ':reference' => $reference, ':data' => $data, ':description' => $description, ':vrn' => $vrn, ':type' => $type, ':id' => $id);
           $query->execute($parameters);
        }
        //or add a new  record
        else{
           $sql="INSERT INTO repairschemes (id,vehicleid,title,reference,data,description,vrn,type) VALUES (NULL,:vehicleid,:title,:reference,:data,:description,:vrn,:type)";
           $query = $database->prepare($sql);
           $parameters=array(':vehicleid' => $vehicleid,':title' => $title, ':reference' => $reference, ':data' => $data, ':description' => $description, ':vrn' => $vrn, ':type' => $type);
            //echo '<br><br><br>[ PDO DEBUG ]: ' . Helper::debugPDO($sql, $parameters);  exit();
           $query->execute($parameters);
           $id=$database->lastInsertId();
        }
        $results=self::getRepairSchemes($vehicleid,$id);
        return $results;

     }

     /**
     * delete a repair scheme record
     * parameters id(int)
     * returns array
     */
     public static function deleteRepairSchemeRecord($id,$vehicleid)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        if($id>0){
           $sql="DELETE FROM repairschemes WHERE id=:id;";
           $query = $database->prepare($sql);
           $parameters=array(':id' => $id);
           $query->execute($parameters);
        }
        $results=self::getRepairSchemes($vehicleid);
        return $results;
     }

     /**
     * gets a role for a given id
     * parameters id(int)
     * returns array
     */
     public static function getRole($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM roles WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters = array(':id' => $id);
        $query->execute($parameters);
        return $query->fetch(PDO::FETCH_ASSOC);

     }

     /**
     * gets a list of all roles
     * parameters none
     * returns array
     */
     public static function getRoleList()
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT id,role FROM roles";
        $query = $database->prepare($sql);
        $query->execute();
        return $query->fetchAll(PDO::FETCH_ASSOC);
     }

     /**
     * save a role
     * parameters id(int),role(string),permissions(string)
     * returns array
     */
     public static function saveRole($id,$role,$permissions)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $results=array();
        if($id){
           if($id>0){
              $sql="UPDATE roles SET role=:role,permissions=:permissions WHERE id=:id";
              $query = $database->prepare($sql);
              $parameters = array( ':role' => $role,':permissions' => $permissions,':id' => $id);
              $query->execute($parameters);
           }
           else{
              $sql="INSERT INTO roles(id, role, permissions) VALUES (null,:role,:permissions)";
              $query = $database->prepare($sql);
              $parameters = array(':role' => $role,':permissions' => $permissions);
              $query->execute($parameters);
              $id = $database->lastInsertId();
           }
        }
        $results['selected']=$id;
        $results['roles']=self::getRoleList();
        return $results;
     }

     /**
     * delete a role
     * parameters id(int)
     * returns array
     */
     public static function deleteRole($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="DELETE FROM roles WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters = array(':id' => $id);
        $query->execute($parameters);
        return self::getRoleList();
     }

     /**
     * register a new username
     * parameters proposedusername(string)
     * returns array
     */
     public static function registerUserName($proposedusername)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $confirmedusername="";
        $results=array();
        $token=base64_encode( openssl_random_pseudo_bytes(32));
        //first check username doesn't already exist
        $sql="SELECT username FROM users WHERE username LIKE :proposedusername";
        $query = $database->prepare($sql);
        $parameters = array(':proposedusername' => $proposedusername);
        $query->execute($parameters);
        $usernames=$query->fetchAll();
        if(sizeof($usernames)>0){
          $usernameindex=1;
          $indices=array();
          foreach ($usernames as $username){
             $indices=filter_var($username, FILTER_SANITIZE_NUMBER_INT);
          }
          //TODO
        }
        else{
           $username=$proposedusername;
        }

        $sql="INSERT INTO users (id,username,token,role) VALUES (null,:username,:token,-1)";
        $query = $database->prepare($sql);
        $parameters = array(':username' => $username, ':token' => $token);
        $query->execute($parameters);
        $results['username']=$username;
        $results['token']=urlencode($token);
        return $results;

     }

     /**
     * get existing users
     * parameters none
     * returns array
     */
     public static function getUsers()
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM users";
        $query = $database->prepare($sql);
        $query->execute();
        return $query->fetchAll(PDO::FETCH_ASSOC);

     }

     /**
     * open a user record
     * parameters id(int)
     * returns array
     */
     public static function openUserRecord($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="SELECT * FROM users WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters = array(':id' => $id);
        $query->execute($parameters);
        return $query->fetchAll(PDO::FETCH_ASSOC);

     }

     /**
     * save a user record
     * parameters id(int), role(int)
     * returns int
     */
     public static function saveUser($id,$role,$encodeduservehicles)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="UPDATE users SET role=:role,vehicles=:uservehicles WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters = array(':id' => $id, ':role' => $role, ':uservehicles' => $encodeduservehicles);
        $query->execute($parameters);
        return array('id' => $id);

     }

     /**
     * delete a user
     * parameters id(int)
     * returns array
     */
     public static function deleteUser($id)
     {
        $database = DatabaseFactory::getFactory()->getConnection();

        $sql="DELETE FROM users WHERE id=:id";
        $query = $database->prepare($sql);
        $parameters = array(':id' => $id);
        $query->execute($parameters);
        return self::getUsers();
     }


}